#include "CameraCBuffer.hlsli"
#include "SamplerStates.hlsli"
#include "Common.hlsli"

Texture2D SourceTex : register(t0);
Texture1D<float> SagittalTex : register(t1);
Texture1D<float> VignettingTex : register(t2);

struct PSInput {
	float4 PositionViewportS : SV_Position;
	float2 TexCoords : TEXCOORD0;
};

// Num samples per blur direction (+ and - sagittal, + and - tangential)
// i.e. additional samples per pass = numSamples * 2;
static const int numSamples = 4;

float4 main(PSInput input) : SV_TARGET
{
	// Convert from texture space [0, 1] to film space [-FilmDimensions/2, FilmDimensions/2]
	float x = (input.TexCoords.x - 0.5) * FilmWidth;
	float y = (input.TexCoords.y - 0.5) * FilmHeight;
	float radius = sqrt(x * x + y * y);
	float coord1D = radius / (22 * kMmToMeters); // Convert from range of [0, 22] to [0, 1]

	// Astigmatism - sagittal blur
	float sagittalFactor = 0.01 * SagittalTex.SampleLevel(LinearSampler, coord1D, 0);

	// Sample the source image.
	float4 color = SourceTex.SampleLevel(BorderedLinearSampler, input.TexCoords, 0);
	int sampleCount = 1;

	// Sample the color buffer along the blur vector.
	if(sagittalFactor != 0) {
		float sagittalDeltaFactor = sagittalFactor / (numSamples * 2);
		// Sagittal vector in texture space
		float2 sagittalVector = float2(x / FilmWidth, y / FilmHeight);
		for(int i = 1; i <= numSamples; ++i)  
		{
			// Sagittal samples
			float2 offset = i * sagittalDeltaFactor * sagittalVector;
			color += SourceTex.SampleLevel(BorderedLinearSampler, input.TexCoords + offset, 0);
			color += SourceTex.SampleLevel(BorderedLinearSampler, input.TexCoords - offset, 0);
		}
		sampleCount += numSamples * 2;
	}
	
	// Average all of the samples to get the final blur color.
	color /= sampleCount;

	// Vignetting
	float vignetting = 0.01 * VignettingTex.SampleLevel(LinearSampler, coord1D, 0);
	float4 finalColor = color * vignetting;

	return finalColor;
}